<?php
if (!defined('ABSPATH')) { exit; }

function smseo_generate_meta_title($title, $content) {
    $title = trim($title);
    if ($title !== '') { return $title; }
    $firstSentence = preg_split('/[\.!?]/', trim($content));
    $candidate = trim($firstSentence[0] ?? '');
    return $candidate !== '' ? mb_substr($candidate, 0, 60) : 'Article';
}

function smseo_generate_meta_description($content) {
    $text = trim(preg_replace('/\s+/', ' ', strip_tags($content)));
    return mb_substr($text, 0, 160);
}

function smseo_extract_keywords($content) {
    $text = strtolower(strip_tags($content));
    $text = preg_replace('/[^a-z0-9\s]/', ' ', $text);
    $words = preg_split('/\s+/', $text, -1, PREG_SPLIT_NO_EMPTY);
    $stop = ['the','a','an','and','or','of','to','in','on','for','with','at','by','is','it','this','that'];
    $freq = [];
    foreach ($words as $w) {
        if (mb_strlen($w) < 3) continue;
        if (in_array($w, $stop, true)) continue;
        $freq[$w] = ($freq[$w] ?? 0) + 1;
    }
    arsort($freq);
    return array_slice(array_keys($freq), 0, 20);
}

function smseo_flesch_score($content) {
    $text = strip_tags($content);
    $sentences = preg_split('/[\.!?]+/', $text, -1, PREG_SPLIT_NO_EMPTY);
    $words = preg_split('/\s+/', trim($text), -1, PREG_SPLIT_NO_EMPTY);
    $syllables = 0;
    foreach ($words as $w) { $syllables += smseo_count_syllables($w); }
    $numSent = max(1, count($sentences));
    $numWords = max(1, count($words));
    $score = 206.835 - (1.015 * ($numWords / $numSent)) - (84.6 * ($syllables / $numWords));
    return round($score, 2);
}

function smseo_count_syllables($word) {
    $word = strtolower(preg_replace('/[^a-z]/', '', $word));
    if ($word === '') return 0;
    $word = preg_replace('/e$/', '', $word);
    $groups = preg_match_all('/[aeiouy]+/', $word);
    return max(1, (int)$groups);
}

function smseo_count_missing_alts($html) {
    $c = 0;
    if (preg_match_all('/<img\b[^>]*>/i', $html, $m)) {
        foreach ($m[0] as $img) {
            $hasAlt = preg_match('/\balt\s*=\s*"[^"]*"/i', $img);
            if (!$hasAlt) $c++;
        }
    }
    return $c;
}

function smseo_compute_score($data) {
    $score = 0;
    if (!empty($data['metaTitle'])) $score += 20;
    if (!empty($data['metaDesc']) && mb_strlen($data['metaDesc']) >= 100) $score += 15; else if (!empty($data['metaDesc'])) $score += 10;
    $kwCount = is_array($data['keywords']) ? count($data['keywords']) : 0;
    if ($kwCount >= 5) $score += 20; else if ($kwCount >= 3) $score += 10;
    $read = (float)$data['readability']; if ($read >= 60) $score += 20; else if ($read >= 40) $score += 10;
    $alts = (int)$data['missingAlts']; if ($alts === 0) $score += 10; else if ($alts <= 2) $score += 5;
    $contentLen = mb_strlen($data['content'] ?? ''); if ($contentLen >= 400) $score += 15; else if ($contentLen >= 200) $score += 8;
    return min(100, $score);
}

function smseo_get_site_summary() {
    $count_posts = wp_count_posts('post');
    $idx = (int)($count_posts->publish ?? 0) + (int)wp_count_posts('page')->publish;
    $avg = 0; $n = 0;
    $recent = get_posts(['post_type'=>['post','page'],'posts_per_page'=>20,'post_status'=>'publish']);
    foreach ($recent as $p) { $avg += (int)get_post_meta($p->ID, '_smseo_score', true); $n++; }
    return ['indexed_estimate'=>$idx, 'avg_score'=>$n? round($avg/$n,1):0];
}

function smseo_scan_broken_links() {
    $broken = [];
    $recent = get_posts(['post_type'=>['post','page'],'posts_per_page'=>50,'post_status'=>'publish']);
    foreach ($recent as $p) {
        $content = $p->post_content;
        if (preg_match_all('#https?://[^\s\)\"]+#', $content, $m)) {
            foreach (array_slice($m[0],0,50) as $u) {
                $resp = wp_remote_head($u, ['timeout'=>4]);
                $code = is_wp_error($resp) ? 'ERR' : wp_remote_retrieve_response_code($resp);
                if ($code >= 400 || $code === 0) { $broken[] = ['url'=>$u, 'status'=>$code]; }
            }
        }
    }
    return $broken;
}

function smseo_regenerate_sitemap() {
    $posts = get_posts(['post_type'=>['post','page'],'post_status'=>'publish','numberposts'=>-1]);
    $xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    $xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
    $home = home_url('/');
    $xml .= "  <url><loc>" . esc_url($home) . "</loc><changefreq>daily</changefreq></url>\n";
    foreach ($posts as $p) {
        $loc = get_permalink($p);
        $xml .= "  <url><loc>" . esc_url($loc) . "</loc><changefreq>weekly</changefreq></url>\n";
    }
    $xml .= "</urlset>\n";
    $upload_dir = wp_upload_dir();
    $path = trailingslashit($upload_dir['basedir']) . 'sm-seo-sitemap.xml';
    file_put_contents($path, $xml);
}

function smseo_indexnow_ping($url) {
    $key = get_option('smseo_indexnow_key', '');
    if ($key === '') return;
    $site = home_url('/');
    $endpoint = 'https://api.indexnow.org/indexnow';
    $body = wp_json_encode(['host'=>parse_url($site, PHP_URL_HOST),'key'=>$key,'keyLocation'=>$site . $key . '.txt','urlList'=>[$url]]);
    wp_remote_post($endpoint, [
        'headers' => ['Content-Type'=>'application/json'],
        'body' => $body,
        'timeout' => 5
    ]);
}