<?php
/*
Plugin Name: SM Auto Alt Generator
Plugin URI: https://smarttoolhub.digital/
Description: Automatically generates missing image ALT text from filenames and related page content with optional AI-style captions, bulk fixer, upload-time alt setting, and SEO coverage report.
Version: 1.0.0
Author: SmartToolHub
Author URI: https://smarttoolhub.digital/
License: GPLv2 or later
Text Domain: sm-auto-alt-generator
*/
if (!defined('ABSPATH')) { exit; }

define('SMALTAI_VER','1.0.0');
define('SMALTAI_FILE', __FILE__);
define('SMALTAI_DIR', plugin_dir_path(__FILE__));

require_once SMALTAI_DIR . 'includes/util.php';

function smaltai_is_image($post_id){ $p = get_post($post_id); if (!$p) return false; $mime = get_post_mime_type($post_id); return $p->post_type==='attachment' && is_string($mime) && strpos($mime,'image/')===0; }

function smaltai_related_content($attachment_id){
    $pid = (int) get_post_field('post_parent', $attachment_id);
    $title = '';
    $content = '';
    if ($pid > 0) {
        $post = get_post($pid);
        if ($post) { $title = (string)$post->post_title; $content = (string)$post->post_content; }
    } else {
        $url = wp_get_attachment_url($attachment_id);
        if ($url) {
            global $wpdb;
            $like = '%' . $wpdb->esc_like($url) . '%';
            $row = $wpdb->get_row($wpdb->prepare("SELECT ID, post_title, post_content FROM {$wpdb->posts} WHERE post_status='publish' AND post_type IN ('post','page','product') AND post_content LIKE %s LIMIT 1", $like));
            if ($row) { $title = (string)$row->post_title; $content = (string)$row->post_content; }
        }
    }
    return [$title,$content];
}

function smaltai_generate_alt($attachment_id){
    $use_ai = get_option('smaltai_enable_ai','0') === '1';
    $max_len = max(20, min(140, (int)get_option('smaltai_max_len', 100)));
    $file = get_attached_file($attachment_id);
    $base = $file ? basename($file) : sanitize_title(get_the_title($attachment_id));
    $base = strtolower(preg_replace('/\.[a-z0-9]+$/','', $base));
    $base = preg_replace('/[_\-]+/',' ', $base);
    $base = preg_replace('/\s+/',' ', trim($base));
    list($ptitle,$pcontent) = smaltai_related_content($attachment_id);
    $text = trim($ptitle . ' ' . $pcontent . ' ' . $base);
    $kws = smaltai_keywords($text);
    $top = array_slice($kws, 0, 6);
    $phrase = implode(' ', $top);
    if ($use_ai) {
        $prefix = $ptitle ? ('image of ' . strtolower($ptitle) . ' ') : 'illustration of ';
        $phrase = trim($prefix . $phrase);
    }
    $phrase = smaltai_titlecase($phrase);
    if ($phrase === '') { $phrase = smaltai_titlecase($base); }
    $phrase = substr($phrase, 0, $max_len);
    return trim($phrase);
}

function smaltai_set_alt($attachment_id, $alt){ if ($alt !== '') { update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt); } }

function smaltai_on_upload($post_id){ if (!smaltai_is_image($post_id)) return; $alt = get_post_meta($post_id, '_wp_attachment_image_alt', true); if ($alt==='') { $g = smaltai_generate_alt($post_id); smaltai_set_alt($post_id, $g); } }
add_action('add_attachment','smaltai_on_upload');

function smaltai_media_stats(){
    $q = new WP_Query(['post_type'=>'attachment','post_status'=>'inherit','posts_per_page'=>-1,'fields'=>'ids','post_mime_type'=>'image']);
    $ids = $q->posts;
    $total = count($ids);
    $with = 0; $without = 0;
    foreach ($ids as $id){ $alt = get_post_meta($id,'_wp_attachment_image_alt',true); if (is_string($alt) && trim($alt)!=='') $with++; else $without++; }
    $pct = $total>0 ? round(($with/$total)*100, 1) : 0.0;
    return [$total,$with,$without,$pct];
}

function smaltai_fix_all_missing(){
    $fixed = 0;
    $q = new WP_Query(['post_type'=>'attachment','post_status'=>'inherit','posts_per_page'=>-1,'fields'=>'ids','post_mime_type'=>'image']);
    foreach ($q->posts as $id){ $alt = get_post_meta($id,'_wp_attachment_image_alt',true); if (!is_string($alt) || trim($alt)==='') { $g = smaltai_generate_alt($id); if ($g!=='') { smaltai_set_alt($id,$g); $fixed++; } } }
    return $fixed;
}

add_action('admin_menu', function(){
    add_menu_page('SM Auto Alt', 'SM Auto Alt', 'manage_options', 'sm-auto-alt', function(){
        if (!current_user_can('manage_options')) return;
        if (isset($_POST['smaltai_save']) && check_admin_referer('smaltai_save')) {
            update_option('smaltai_enable_ai', isset($_POST['smaltai_enable_ai']) ? '1' : '0');
            $ml = (int)($_POST['smaltai_max_len'] ?? '100');
            update_option('smaltai_max_len', max(20, min(140, $ml)));
            echo '<div class="notice notice-success"><p>Settings saved.</p></div>';
        }
        if (isset($_POST['smaltai_fix_all']) && check_admin_referer('smaltai_fix')) {
            $n = smaltai_fix_all_missing();
            echo '<div class="notice notice-success"><p>Fixed ' . esc_html($n) . ' images.</p></div>';
        }
        list($total,$with,$without,$pct) = smaltai_media_stats();
        $enable_ai = get_option('smaltai_enable_ai','0')==='1';
        $max_len = (int)get_option('smaltai_max_len',100);
        echo '<div class="wrap"><h1>SM Auto Alt Generator</h1>';
        echo '<h2>SEO Report</h2>';
        echo '<p>Total images: <strong>' . esc_html($total) . '</strong></p>';
        echo '<p>With ALT: <strong>' . esc_html($with) . '</strong></p>';
        echo '<p>Missing ALT: <strong>' . esc_html($without) . '</strong></p>';
        echo '<p>Coverage: <strong>' . esc_html($pct) . '%</strong></p>';
        echo '<h2>Settings</h2>';
        echo '<form method="post">';
        wp_nonce_field('smaltai_save');
        echo '<label><input type="checkbox" name="smaltai_enable_ai" ' . ($enable_ai?'checked':'') . ' /> Enable AI caption generator</label><br/>';
        echo '<label>Max ALT length <input type="number" name="smaltai_max_len" value="' . esc_attr($max_len) . '" min="20" max="140" /></label>';
        echo '<p><button class="button button-primary" name="smaltai_save" value="1">Save</button></p>';
        echo '</form>';
        echo '<h2>Bulk Actions</h2>';
        echo '<form method="post">';
        wp_nonce_field('smaltai_fix');
        echo '<p><button class="button" id="smaltai-fix" name="smaltai_fix_all" value="1">Fix All Missing ALT Tags</button></p>';
        echo '</form>';
        echo '<script>document.getElementById("smaltai-fix").addEventListener("click",function(){this.disabled=true;});</script>';
        echo '</div>';
    }, 'dashicons-tag', 87);
});

?>